"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IconIcns = void 0;
var _packbits = require("@fiahfy/packbits");
var _icon = require("../icon.js");
const typeArgb = ['ic04', 'ic05'];
const typePng = ['icp4', 'icp5', 'icp6', 'ic07', 'ic08', 'ic09', 'ic10', 'ic11', 'ic12', 'ic13', 'ic14'];
const typeIcon24Bit = ['is32', 'il32', 'ih32', 'it32'];
const typeMask8Bit = ['s8mk', 'l8mk', 'h8mk', 't8mk'];

/**
 * Icon entry.
 */

/**
 * IconIcns object.
 */
class IconIcns extends _icon.Icon {
  /**
   * Option to include TOC tag (table of contents) in encode.
   */
  toc = false;

  /**
   * List of icon entries.
   */
  entries = [];

  /**
   * Types that are ARGB.
   */
  _typeArgb = new Set(typeArgb);

  /**
   * Types that are PNG.
   */
  _typePng = new Set(typePng);

  /**
   * Types that are icon 24-bit.
   */
  _typeIcon24Bit = new Set(typeIcon24Bit);

  /**
   * Types that are mask 8-bit.
   */
  _typeMask8Bit = new Set(typeMask8Bit);

  /**
   * IconIcns constructor.
   */
  constructor() {
    super();
  }

  /**
   * Add an icon from PNG data.
   *
   * @param data PNG data.
   * @param types Types to encode as.
   * @param raw Use raw PNG data without re-encoding for the PNG types.
   */
  addFromPng(data, types, raw = false) {
    if (!raw) {
      this.addFromRgba(this._decodePngToRgba(data), types);
      return;
    }
    let rgba = null;
    for (const type of types) {
      if (this._typePng.has(type)) {
        this.entries.push({
          type,
          data: Buffer.concat([data], data.length)
        });
        continue;
      }
      rgba || (rgba = this._decodePngToRgba(data));
      this.addFromRgba(rgba, [type]);
    }
  }

  /**
   * Add an icon from RGBA image data.
   *
   * @param imageData RGBA image data.
   * @param types Types to encode as.
   */
  addFromRgba(imageData, types) {
    for (const type of types) {
      this._addFromRgbaType(imageData, type);
    }
  }

  /**
   * Encode icon.
   *
   * @returns Encoded icon.
   */
  encode() {
    const {
      toc
    } = this;
    const head = Buffer.alloc(8);
    head.write('icns', 0);
    let size = 8;
    let tocSize = 8;
    const tocHead = toc ? Buffer.from('TOC ----') : null;
    const tocs = tocHead ? [tocHead] : [];
    const images = [];
    for (const {
      type,
      data
    } of this.entries) {
      const tagType = Buffer.alloc(4);
      tagType.write(type, 0);
      const tagSize = Buffer.alloc(4);
      const tagSizeValue = data.length + 8;
      tagSize.writeUInt32BE(tagSizeValue, 0);
      if (toc) {
        tocs.push(tagType, tagSize);
        tocSize += 8;
        size += 8;
      }
      images.push(tagType, tagSize, data);
      size += tagSizeValue;
    }
    if (tocHead) {
      tocHead.writeUInt32BE(tocSize, 4);
      size += 8;
    }
    head.writeUInt32BE(size, 4);
    return Buffer.concat([head, ...tocs, ...images], size);
  }

  /**
   * Add an icon from RGBA image data, individual type.
   *
   * @param imageData RGBA image data.
   * @param type Type to encode as.
   */
  _addFromRgbaType(imageData, type) {
    if (this._typeArgb.has(type)) {
      this.entries.push({
        type,
        data: this._encodeRgbaToTypeArgb(imageData, type)
      });
      return;
    }
    if (this._typePng.has(type)) {
      this.entries.push({
        type,
        data: this._encodeRgbaToTypePng(imageData, type)
      });
      return;
    }
    if (this._typeIcon24Bit.has(type)) {
      this.entries.push({
        type,
        data: this._encodeRgbaToTypeIcon24Bit(imageData, type)
      });
      return;
    }
    if (this._typeMask8Bit.has(type)) {
      this.entries.push({
        type,
        data: this._encodeRgbaToTypeMask8Bit(imageData, type)
      });
      return;
    }
    throw new Error(`Unknown type: ${type}`);
  }

  /**
   * Encode RGBA image data to ARGB.
   *
   * @param imageData RGBA image data.
   * @param _type Icon type.
   * @returns Encoded data.
   */
  _encodeRgbaToTypeArgb(imageData, _type) {
    // The compressed data always has an ARGB header.
    return this._encodeRgbaToPackBits(imageData, true, Buffer.from('ARGB', 'ascii'));
  }

  /**
   * Encode RGBA image data to PNG.
   *
   * @param imageData RGBA image data.
   * @param _type Icon type.
   * @returns Encoded data.
   */
  _encodeRgbaToTypePng(imageData, _type) {
    return this._encodeRgbaToPng(imageData);
  }

  /**
   * Encode RGBA image data to icon 24-bit.
   *
   * @param imageData RGBA image data.
   * @param type Icon type.
   * @returns Encoded data.
   */
  _encodeRgbaToTypeIcon24Bit(imageData, type) {
    // The 'it32' type has 4 null byte header.
    return this._encodeRgbaToPackBits(imageData, false, type === 'it32' ? Buffer.alloc(4) : null);
  }

  /**
   * Encode RGBA image data to mask 8-bit.
   *
   * @param imageData RGBA image data.
   * @param _type Icon type.
   * @returns Encoded data.
   */
  _encodeRgbaToTypeMask8Bit(imageData, _type) {
    return this._encodeRgbaChannel(imageData, 3);
  }

  /**
   * Encode RGBA image data to packbits.
   *
   * @param imageData RGBA image data.
   * @param alpha Incldue the alpha channel.
   * @param header Header to prepend to the output.
   * @returns Encoded data.
   */
  _encodeRgbaToPackBits(imageData, alpha, header = null) {
    const pieces = header ? [header] : [];
    if (alpha) {
      // A:
      pieces.push(this._encodePackBitsIcns(this._encodeRgbaChannel(imageData, 3)));
    }
    // RGB:
    pieces.push(this._encodePackBitsIcns(this._encodeRgbaChannel(imageData, 0)), this._encodePackBitsIcns(this._encodeRgbaChannel(imageData, 1)), this._encodePackBitsIcns(this._encodeRgbaChannel(imageData, 2)));
    return Buffer.concat(pieces);
  }

  /**
   * Encode channel from RGBA image data.
   *
   * @param imageData RGBA image data.
   * @param index Channel index (R=0, B=1, G=2, A=3).
   * @returns Encoded data.
   */
  _encodeRgbaChannel(imageData, index) {
    const {
      data
    } = imageData;
    const size = data.length;
    const encoded = Buffer.alloc(size / 4);
    for (let i = index, j = 0; i < size; i += 4) {
      encoded.writeUInt8(data[i], j++);
    }
    return encoded;
  }

  /**
   * Encode data using PackBits ICNS compression.
   *
   * @param data Data to be compressed.
   * @returns Compressed data.
   */
  _encodePackBitsIcns(data) {
    return (0, _packbits.encode)(data, {
      format: 'icns'
    });
  }
}
exports.IconIcns = IconIcns;
//# sourceMappingURL=icns.js.map
